﻿using System.Collections.Generic;

namespace Hims.Api.Controllers
{
    using Domain.Configurations;
    using Domain.Helpers;
    using Domain.Services;
    using Hims.Api.Models;
    using Hims.Api.Models.GynEncounter;
    using Hims.Domain.Repositories.UnitOfWork;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Newtonsoft.Json;
    using Newtonsoft.Json.Serialization;
    using Shared.DataFilters;
    using Shared.EntityModels;
    using Shared.Library;
    using Shared.Library.Enums;
    using Shared.UserModels;
    using Shared.UserModels.Filters;
    using System;
    using System.Linq;
    using System.Threading.Tasks;
    using Utilities;

    /// <summary>
    /// The internal medicine controller.
    /// </summary>
    [Authorize]
    [Route("api/gyn-encounter")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class GynEncounterController : BaseController
    {
        /// <summary>
        /// the provider service
        /// </summary>
        private readonly IEncounterService encounterServices;

        /// <summary>
        /// The push notification helper.
        /// </summary>
        private readonly IPushNotificationHelper pushNotificationHelper;

        /// <summary>
        /// The account session services.
        /// </summary>
        private readonly IAccountSessionService accountSessionServices;


        /// <summary>
        /// The account session services.
        /// </summary>
        private readonly IGynEncounterService gynEncounterService;


        /// <summary>
        /// the aes helper
        /// </summary>
        private readonly IAESHelper aesHelper;

        /// <summary>
        /// The appointments services.
        /// </summary>
        private readonly IAppointmentService appointmentsServices;

        /// <summary>
        /// The account service.
        /// </summary>
        private readonly IAccountService accountService;

        /// <summary>
        /// The amazon s3 helper.
        /// </summary>
        private readonly IDocumentHelper documentHelper;

        /// <summary>
        /// The configuration.
        /// </summary>
        private readonly IAmazonS3Configuration configuration;
        /// <summary>
        /// The unit of work.
        /// </summary>
        private readonly IUnitOfWork unitOfWork;

        /// <inheritdoc />
        public GynEncounterController(
            IGynEncounterService gynEncounterService,
            IAppointmentService appointmentsServices,
            IAESHelper aesHelper,
            IDocumentHelper documentHelper,
            IEncounterService encounterServices,
            IPushNotificationHelper pushNotificationHelper,
            IAccountSessionService accountSessionServices,
            IAmazonS3Configuration configuration,
            IAccountService accountService, IUnitOfWork unitOfWork)
        {
            this.gynEncounterService = gynEncounterService;
            this.appointmentsServices = appointmentsServices;
            this.aesHelper = aesHelper;
            this.documentHelper = documentHelper;
            this.configuration = configuration;
            this.accountService = accountService;
            this.encounterServices = encounterServices;
            this.pushNotificationHelper = pushNotificationHelper;
            this.accountSessionServices = accountSessionServices;
            this.unitOfWork = unitOfWork;
        }

        /// <summary>
        /// The find gyn encounter.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("find")]
        [ProducesResponseType(typeof(EncounterResource), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FindAsync([FromBody] GynEncounterFilterModel model)
        {
            model = (GynEncounterFilterModel)EmptyFilter.Handler(model);
            var appointmentId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedAppointmentId));
            var gynEncounter = await this.gynEncounterService.FindAsync(appointmentId, model.IsAdmission);
            var providerId = !string.IsNullOrEmpty(model.EncryptedProviderId)
                ? Convert.ToInt32(this.aesHelper.Decode(model.EncryptedProviderId))
                : 0;
            var patientId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedPatientId));
            var appointmentList = new List<AppointmentModel>();
            var oldAppointment = new PreviousAppointmentModel();

            if (!model.IsAdmission)
            {
                var appointments = await this.appointmentsServices.FindByPatientAsync(patientId, providerId);
                appointmentList = appointments.ToList();

                foreach (var item in appointmentList)
                {
                    item.EncryptedAppointmentId = this.aesHelper.Encode(item.AppointmentId.ToString());
                    item.EncryptedPatientId = model.EncryptedPatientId;
                    item.AppointmentTimeString = Convert.ToDateTime(DateTime.Now.ToString("yyyy-MM-dd"))
                        .Add(item.AppointmentTime).ToString("hh:mm tt");
                }
            }

            return this.Success(new { Dashboard = gynEncounter, Appointments = appointmentList, OldAppointment = oldAppointment });
        }

        /// <summary>
        /// The find gyn encounter.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("find-encounter")]
        [ProducesResponseType(typeof(EncounterResource), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FindEncounterDataAsync([FromBody] GynEncounterFilterModel model)
        {
            model = (GynEncounterFilterModel)EmptyFilter.Handler(model);
            var appointmentId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedAppointmentId));
            var gynEncounter = await this.gynEncounterService.FindEncounterDataAsync(appointmentId, model.IsAdmission);
            var providerId = !string.IsNullOrEmpty(model.EncryptedProviderId)
                ? Convert.ToInt32(this.aesHelper.Decode(model.EncryptedProviderId))
                : 0;
            var patientId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedPatientId));
            var appointmentList = new List<AppointmentModel>();
            var oldAppointment = new PreviousAppointmentModel();

            if (!model.IsAdmission)
            {
                var appointments = await this.appointmentsServices.FindByPatientAsync(patientId, providerId);
                appointmentList = appointments.ToList();

                foreach (var item in appointmentList)
                {
                    item.EncryptedAppointmentId = this.aesHelper.Encode(item.AppointmentId.ToString());
                    item.EncryptedPatientId = model.EncryptedPatientId;
                    item.AppointmentTimeString = Convert.ToDateTime(DateTime.Now.ToString("yyyy-MM-dd"))
                        .Add(item.AppointmentTime).ToString("hh:mm tt");
                }
            }

            return this.Success(new { Dashboard = gynEncounter, Appointments = appointmentList, OldAppointment = oldAppointment });
        }
        /// <summary>
        /// The modify async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("modify")]
        [ProducesResponseType(typeof(int), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> ModifyAsync([FromBody] GynEncounterModifyModel model)
        {
            model = (GynEncounterModifyModel)EmptyFilter.Handler(model);
            int aptId = await this.gynEncounterService.getApppointmentId(model.AppointmentId, model.IsAdmission);
            var gynEncounterId = aptId == 0 ? await this.gynEncounterService.AddAltAsync(model) : await this.gynEncounterService.UpdateAltAsync(model);
            if ((gynEncounterId !=null) && (model.Type==GynEncounterType.GynHistory))
            {
                CommonEncounterModel commonModel=new CommonEncounterModel();
                commonModel.CommonEncounterId =(int) model.CommonEncounterId;
                commonModel.GPLA = model.JsonString;
                commonModel.AppointmentId = model.AppointmentId;
                commonModel.CreatedBy = model.ModifiedBy;
                commonModel.GynEncounterType = model.Type;
                await InserCommonEncounter(commonModel);
            }
           
            switch (gynEncounterId.Response)
            {
                case 0:
                    return this.ServerError();
                default:                    
                    var basicDetails = await this.encounterServices.GetBasicAppointmentDetails(model.AppointmentId, model.IsAdmission);
                    await this.appointmentsServices.UpdateEncounterTypeAsync(model.AppointmentId,
                        (int)EncounterTypes.GynEncounter, model.IsAdmission);
                    return this.Success(gynEncounterId.Response);
            }
        }

        /// <summary>
        /// To find gyn encounter dashboard
        /// </summary>
        /// <param name="model" >
        /// The encounter filter model.
        /// </param>
        /// <returns>
        /// The encounter model.
        /// </returns>
        [HttpPost]
        [Route("find-dashboard")]
        [ProducesResponseType(typeof(EncounterResource), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FindDashboardAsync([FromBody] GynEncounterFilterModel model)
        {
            try
            {
                model = (GynEncounterFilterModel)EmptyFilter.Handler(model);
                var appointmentId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedAppointmentId));
                var response = await this.gynEncounterService.FindDashboardAsync(appointmentId, model.Type, model.IsAdmission);
                if (response != null)
                {
                    response.EncryptedAppointmentId = this.aesHelper.Encode(appointmentId.ToString());
                }
                return this.Success(response);
            }
            catch (Exception e)
            {
                throw;
            }
        }

        /// <summary>
        /// The save encounter images.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("gyn-save-skin")]
        public async Task<ActionResult> SaveEncounterImages([FromBody] GynEncounterImages model)
        {
            model = (GynEncounterImages)EmptyFilter.Handler(model);
            var appointmentId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedAppointmentId));
            var getPatientAccount = await this.gynEncounterService.GetAccountBasedOnAppointmentId(appointmentId, model.IsAdmission);
            var guid = await this.accountService.FindGuidAsync(getPatientAccount.PatientId, Roles.Patient);

            foreach (var image in model.Images)
            {
                if (image.Base64String.Contains(this.configuration.BucketURL) || image.Base64String.Contains("assets"))
                {
                    continue;
                }

                string appointment;

                try
                {
                    appointment = getPatientAccount.AppointmentDate.Add(getPatientAccount.AppointmentTime)
                        .ToString("yyyy-MM-dd_hh:mm_tt");
                }
                catch
                {
                    appointment = getPatientAccount.AppointmentNo;
                }

                var folderName = $@"gynEncounter/{appointment}/PhysicalExam/{model.Type}";
                var url = await this.documentHelper.UploadEncounterImagesAsync(image.Base64String, guid, folderName,
                    image.Type);
                image.Base64String = url;
            }

            var finalModel = new GynEncounterModifyModel
            {
                AppointmentId = appointmentId,
                GynEncounterId = model.GynEncounterId ?? 0,
                JsonString = JsonConvert.SerializeObject(model,
                    new JsonSerializerSettings { ContractResolver = new CamelCasePropertyNamesContractResolver() }),
                Type = model.Type,
                ModifiedBy = model.ModifiedBy
            };
            return await this.ModifyAsync(finalModel);
        }

        /// <summary>
        /// The find full transcript async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("find-full-transcript")]
        [ProducesResponseType(typeof(GynEncounterFullTranscriptModel), 200)]
        [ProducesResponseType(500)]
        [AllowAnonymous]
        public async Task<ActionResult> FindFullTranscriptAsync([FromBody] GynEncounterFilterModel model)
        {
            model = (GynEncounterFilterModel)EmptyFilter.Handler(model);
            var appointmentId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedAppointmentId));

            var gynEncounterFullTranscript = await this.gynEncounterService.FindFullTranscriptAsync(appointmentId,model.IsAdmission);
            if (gynEncounterFullTranscript == null || gynEncounterFullTranscript.GynEncounterId == 0)
            {
                return this.BadRequest("Sorry! We don't have a encounter in the system.");
            }

            var additionalData = await this.encounterServices.GetAdditionalProviderDataAsync(appointmentId);

            if (additionalData != null)
            {
                gynEncounterFullTranscript.Educations = additionalData.Educations;
                gynEncounterFullTranscript.ProviderName = additionalData.ProviderName;
                gynEncounterFullTranscript.ProviderNo = additionalData.ProviderNo;
                gynEncounterFullTranscript.SpecializationName = additionalData.SpecializationName;
                gynEncounterFullTranscript.PracticeName = additionalData.PracticeName;
                gynEncounterFullTranscript.PracticeLocation = additionalData.PracticeLocation;
                gynEncounterFullTranscript.Signature = additionalData.Signature;
                gynEncounterFullTranscript.ClinicPicture = additionalData.ClinicPicture;
            }

            gynEncounterFullTranscript.AppointmentTimeString = Convert
                .ToDateTime(DateTime.Now.ToString("yyyy-MM-dd")).Add(gynEncounterFullTranscript.AppointmentTime)
                .ToString("hh:mm tt");
            return this.Success(gynEncounterFullTranscript);
        }
        /// <summary>
        /// Inserts the anc card asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Authorize]
        [Route("add-gynCardGeneration")]
        public async Task<ActionResult> InsertAsync([FromBody] GYNCardGenerationModel model, [FromHeader] LocationHeader location)
        {
            model = (GYNCardGenerationModel)EmptyFilter.Handler(model);
            var response = await this.gynEncounterService.AddGYNCardAsync(model, location.LocationId);
            return this.Success(response);
        }
        /// <summary>
        /// Fetches all asynchronous.
        /// </summary>
        /// <param name="patientId">The model.</param>
        /// <returns></returns>
        [HttpGet]
        [Route("fetch-gynCard")]
        public async Task<ActionResult> FetchAsync(int patientId)
        {
            var response = await this.gynEncounterService.FetchGynCardAsync(patientId);
            return this.Success(response);
        }
        /// <summary>
        /// To find ob encounter dashboard
        /// </summary>
        /// <param name="model" >
        /// The encounter filter model.
        /// </param>
        /// <returns>
        /// The encounter model.
        /// </returns>
        [HttpPost]
        [Authorize]
        [Route("find-gyn")]
        [ProducesResponseType(typeof(EncounterResource), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FindOb([FromBody] GynEncounterFilterModel model)
        {
            model = (GynEncounterFilterModel)EmptyFilter.Handler(model);
            var appointmentId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedAppointmentId));
            var response = await this.gynEncounterService.FindGyn(appointmentId, model.Type, model.IsAdmission);
            if (response != null)
            {
                response.EncryptedAppointmentId = this.aesHelper.Encode(appointmentId.ToString());
            }
            return this.Success(response);
        }
        [HttpGet]
        [Authorize]
        [Route("find-visitNo")]
        [ProducesResponseType(typeof(EncounterResource), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FindVisitNo(int patientId)
        {
            var response = await this.gynEncounterService.FetchVisitNoAsync(patientId);
            return this.Success(response);
        }
        /// <summary>
        /// Inserts the anc card asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Authorize]
        [Route("add-common-encounter")]
        public async Task<ActionResult> InserCommonEncounter([FromBody] CommonEncounterModel model)
        {
            var records=this.unitOfWork.Appointments.Find(s=>s.AppointmentId== model.AppointmentId);
            int aptId = await this.gynEncounterService.getAptId((int)model.AppointmentId);
             
            var gynEncounterId = aptId == 0 ? await this.gynEncounterService.InsertCommonEncounter(model) : await this.gynEncounterService.UpdateCommonEncounter(model);
            
            if (model.routingValue == "ob-encounter")
            {
                await this.appointmentsServices.UpdateEncounterTypeAsync(model.AppointmentId,
                        (int)EncounterTypes.OBEncounter, false);
            }
            else if (model.routingValue == "gyn-encounter")
            {
                await this.appointmentsServices.UpdateEncounterTypeAsync(model.AppointmentId,
                       (int)EncounterTypes.GynEncounter, false);
            }
            return this.Success(gynEncounterId);
        }

        [HttpPost]
        [Authorize]
        [Route("fetch-common-encounter")]
        public async Task<ActionResult> FindDashboardAsync([FromBody] CommonEncounterModel model)
        {
            try
            {
                model = (CommonEncounterModel)EmptyFilter.Handler(model);
                var appointmentId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedAppointmentId));
                var response = await this.gynEncounterService.FetchCommonEncounter(model.Type,model.GynEncounterType, appointmentId, model.IsAdmission);                
                if (response != null)
                {
                    response.EncryptedAppointmentId = this.aesHelper.Encode(appointmentId.ToString());
                }
                return this.Success(response);
            }
            catch (Exception e)
            {
                throw;
            }
        }

        [HttpGet]
        [Authorize]
        [Route("nurse-note-report")]
        public async Task<ActionResult> FetchNurseNotesReport(string appointmentId)
        {
            try
            {
                var admissionId = Convert.ToInt32(this.aesHelper.Decode(appointmentId));
                var notes = await this.gynEncounterService.FetchCreatedDateNurseNoteReport(admissionId);
                return this.Success(notes);
            }
            catch(Exception e)
            {
                return null;
            }
        }
        [HttpGet]
        [Route("encouter-completed")]
        public async Task<ActionResult> ModifyEncounterStatus(int appointmentId)
        {

            var response = await this.gynEncounterService.ModifyGynEncounterStatus(appointmentId);
            return this.Success(response);
        }

    }

}

